/*
 Whenever FS28 capture image, it will sent RAW image or WSQ image through Bluetooth serial port. Therefore, COM Port should be always listening to recieve the command or WSQ/RAW image data from FS28.
 
 Procedures of COM PORT receving RAW image from FS28:
 Step 1: COM PORT receive the command of downloading RAW image from FS28.
         0x40 0x44 0x00 0x00 0x00 0x00 0x00 0x58 0x02 0x00 0x00 0xDE 0x0D (13Byte command inidcating FS28 start to send RAW image start.)
 Step 2: send ok repond to FS28
         0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xCA, 0x0D (13 bytes respond send to FS28 to reply ok)
 Step 3: COM PORT recieved the 153602 bytes RAW image data and store it into DataReceived[]
 Step 4: send ok repond to FS28
        0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xCA, 0x0D (13 bytes respond send to FS28 to reply ok)
 * 
 *    PC/Phone State                                                                                       FS28 State
 *                                                                           
 *  Command mode                40 44 00 00 00 00 00 58 02 00 00 DE 0D (download RAW image cmd)
 *  Ready              <----------------------------------------------------------------------------  RAW Image Captured
 *  
 *  Command mode                                  40 4A 00 00 00 00 00 00 00 00 40 CA 0D (reply ok)
 *  ready              ---------------------------------------------------------------------------->  RAW Image Captured
 *  
 *  Data mode                                 RAW image data(153602 bytes) byte by byte
 *  receving RAW Image data      <-------------------------------------------------------------------  RAW Image downloading
 *  
 *   Data mode                                 40 4A 00 00 00 00 00 00 00 00 40 CA 0D (reply ok)
 *  complete receving RAW Image   ------------------------------------------------------------------>  RAW Image downloaded

 * --------------------------------------------------------------------------------------------------------------------------------------------------
 Procedures of COM PORT receving WSQ image from FS28:
 Step 1: COM PORT recieve the command of converting WSQ  from FS28
         0x40 0x36 0x4B 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x01 0xC2 0x0D (13Byte command inidcating FS28 is converting RAW image to WSQ image.)
 Step 2: send ok repond to FS28
         0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xCA, 0x0D (13 bytes respond send to FS28 to reply ok)
 Step 3: COM PORT receive the command of downlaoding WSQ from FS28
         0x40 0x0F 0x00 0x00 0x00 0x00 WSQ_size(4bytes) checksum(1byte) 0x0D (13Byte command inidcating FS28 start to send WSQ image start.)
 Step 4: send ok repond to FS28
         0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xCA, 0x0D (13 bytes respond send to FS28 to reply ok)
 Step 5: COM PORT recieved the WSQ image data and store it into DataReceived[]
 Step 6: send ok repond to FS28
         0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xCA, 0x0D (13 bytes respond send to FS28 to reply ok)
 
 *    PC/Phone State                                                                                   FS28 State
 *                                                                           
 *  Command mode                    40 36 4B 00 00 00 00 00 00 00 01 C2 0D (convert WSQ cmd)
 *  Ready              <---------------------------------------------------------------------------  Image Captured
 *  
 *  Command mode                                  40 4A 00 00 00 00 00 00 00 00 40 CA 0D (reply ok)
 *  Ready               --------------------------------------------------------------------------->  Image Captured
 *  
 *  Command mode          40 0F 00 00 00 00 WSQ_size(4bytes) checksum(1byte) 0D (download WSQ cmd)
 *  Ready              <----------------------------------------------------------------------------  WSQ converted
 *  
 *  Command mode                                  40 4A 00 00 00 00 00 00 00 00 40 CA 0D (reply ok)
 *  ready for that WSQ size   --------------------------------------------------------------------------->  WSQ converted
 *  
 *  Data mode                                 WSQ data byte by byte
 *  receving WSQ data      <----------------------------------------------------------------------------  WSQ downloading
 *  
 *   Data mode                                 40 4A 00 00 00 00 00 00 00 00 40 CA 0D (reply ok)
 *  complete receving WSQ   --------------------------------------------------------------------------->  WSQ downloaded
 *  
 *********************************************************************************************************************************/
/*
 Procedures of COM PORT receving fingeprint Sample from FS28:
 Step 1: COM PORT receive the command of downloading sample from FS28.
         0x40 0x4D 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x8D 0x0D (13Byte command inidcating FS28 start to send sample start.)
 Step 2: send ok repond to FS28
         0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xCA, 0x0D (13 bytes respond send to FS28 to reply ok)
 Step 3: COM PORT recieved the 666 bytes sample data and store it into DataReceived[]
 Step 4: send ok repond to FS28
         0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xCA, 0x0D (13 bytes respond send to FS28 to reply ok)
 * 
 *    PC/Phone State                                                                                      FS28 State
 *                                                                           
 *  Command mode                40 4D 00 00 00 00 00 00 00 00 00 8D 0D (download sample cmd)
 *  Ready              <----------------------------------------------------------------------------  Sample created
 *  
 *  Command mode                                  40 4A 00 00 00 00 00 00 00 00 40 CA 0D (reply ok)
 *  ready              ---------------------------------------------------------------------------->   Sample created
 *  
 *  Data mode                                 sample data(666 bytes) byte by byte
 *  receving sample data      <-------------------------------------------------------------------  sample downloading
 *  
 *   Data mode                                 40 4A 00 00 00 00 00 00 00 00 40 CA 0D (reply ok)
 *  complete receving Sample   ------------------------------------------------------------------>  sample download complete
*********************************************************************************************************************************/
#region Namespace Inclusions
using System;
using System.Linq;
using System.Data;
using System.Text;
using System.Drawing;
using System.IO.Ports;
using System.Windows.Forms;
using System.ComponentModel;
using System.Collections.Generic;

using FS28BluetoothMasterModeDemo.Properties;
using System.Threading;
using System.IO;
#endregion

namespace FS28BluetoothMasterModeDemo
{
  #region Public Enumerations
  public enum LogMsgType { Incoming, Outgoing, Normal, Warning, Error };
  #endregion

  public partial class frmTerminal : Form
  {
    #region Local Variables

      private int TotalDataSize = 0; //Record the total size of raw image size or WSQ image size that has to be received, RAW image size = 320x480+2 bytes. WSQ image size not constant, varies with captured image size.
      private bool DataMode = false; //if DataMode == true, that means it is ready to recive the RAW image or WSQ data from FS28. if DataMode == flash, that means it is in Command mode, it receive commands from FS28.
      private int DataReceiveCount = 0;  // for counting how many byte of imag data have been received.
      private byte[] DataReceived = new byte[153602]; //320x480+2, raw image content size is 320 x480 byte , plus 1 byte for checksum and 1 byte for inicating ending(0x0D)
      private int startReceiveTime = 0; // for record the starting time of reciving data.
      private enum ImageTypes { RawImage = 0, WSQ, FutronicSample, AnsiSample,IsoSample };
      private ImageTypes ImageType = ImageTypes.RawImage;
      private bool bImgRxComplete = false; // Flag for indicating finish reciving WSQ or RAW image.
      private byte[] ConvertedRawImage;// for storing the RAW image which is converted from WSQ
      private uint sizeWsqImg = 0;
      private int sizeAnsiIso = 0; //for record the size of ANSI and ISO sample.
      private byte[] Rsp2FS28 = { 0x40, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x20, 0x0D };
      private string[] m_strComPorts;
      private byte[] HostSample = new byte[669];    // for the converted FS28 sample to Host SDK sample. Note: Host SDK sample is 5 bytes more than FS28 sample

    // The main control for communicating through the RS-232 port
    private SerialPort comport = new SerialPort();

    // Various colors for logging info
    private Color[] LogMsgTypeColor = { Color.Blue, Color.Green, Color.Black, Color.Orange, Color.Red };

		private Settings settings = Settings.Default;

      private bool m_bExit = false;

      private delegate void EnableTimerCallback(bool bEnable);

     
    private void EnableTimerInvoke(bool bEnable)
    {
        EnableTimerCallback d = new EnableTimerCallback(this.EnableTimer);
        this.Invoke(d, new object[] { bEnable });
    }

    private void EnableTimer(bool bEnable)
    {
        if (bEnable)
        {
            switch (ImageType)
            {
                case ImageTypes.RawImage:
                    this.ImgRecTmr.Interval = 16000; //16 seconds
                    break;
                case ImageTypes.WSQ:
                    this.ImgRecTmr.Interval = 3000; //3 seconds
                    break;
                case ImageTypes.FutronicSample:
                    this.ImgRecTmr.Interval = 2000; //2 seconds
                    break;
                case ImageTypes.AnsiSample:
                    this.ImgRecTmr.Interval = 2000; //2 seconds
                    break;
                case ImageTypes.IsoSample:
                    this.ImgRecTmr.Interval = 2000; //2 seconds
                    break;
                default:
                    break;

            }
            this.ImgRecTmr.Enabled = true;
            this.btnOpenPort.Visible = false; //not allow user to close the com port when image/WSQ is sending
            this.btnSaveImage.Visible = false; //not allow user to save image when image/WSQ is sending
        }
        else
        {
            this.ImgRecTmr.Enabled = false;
            this.btnOpenPort.Visible = true; //allow user to close the com port after image/WSQ sending.
            this.btnSaveImage.Visible = true; //allow user to save image after image/WSQ sending.
        }
    }

    #endregion

    #region Constructor
    public frmTerminal()
    {	
        // Load user settings
		settings.Reload();

      // Build the form
      InitializeComponent();

      /*******************************************************
       re-size of the size of pictureBox1 and progressBar1
       ******************************************************/
      pictureBox1.Width = 320;
      pictureBox1.Height = 480;
      progressBar1.Width = 192;
      progressBar1.Height = 23;

      /*******************************************************
         re-arrange the position of the btnSaveImage, pictureBox1 and progressBar1
       ******************************************************/
      pictureBox1.Left = rtfTerminal.Right + 10;
      int WidthOffset = btnSaveImage.Right - pictureBox1.Right;
      if (WidthOffset > 0)
      {
          btnSaveImage.Left -= WidthOffset;
      }
      WidthOffset = progressBar1.Left - pictureBox1.Left;
      if (WidthOffset > 0)
      {
          progressBar1.Left -= WidthOffset;
      }
      int HeightOffset = btnSaveImage.Top - pictureBox1.Bottom;
      btnSaveImage.Top = btnSaveImage.Top - HeightOffset + 20;

      HeightOffset = progressBar1.Top - pictureBox1.Bottom;
      progressBar1.Top = progressBar1.Top - HeightOffset + 20;

      WidthOffset = this.Right - pictureBox1.Right;
      if (WidthOffset > 0)
      {
          this.Width = this.Width - WidthOffset + 25;
      }

      if (gbPortSettings.Bottom > progressBar1.Bottom)
                {
                    HeightOffset = this.Bottom - gbPortSettings.Bottom;
                    this.Height = this.Height - HeightOffset + 40;
                }
     
      // Restore the users settings
      InitializeControlValues();

      // Enable/disable controls based on the current state
      EnableControls();


      // When data is recieved through the port, call this method
      comport.ReceivedBytesThreshold = 13;// FS28 will first send the 13 byte command before sending the WSQ or RAW image data
         comport.DataReceived += new SerialDataReceivedEventHandler(port_DataReceived);
    }


    #endregion

    #region Local Methods
    
    /// <summary> Save the user's settings. </summary>
    private void SaveSettings()
    {
			settings.BaudRate = int.Parse(cmbBaudRate.Text);
            if( m_strComPorts.Length > 0 )
			    settings.PortName = m_strComPorts[cmbPortName.SelectedIndex];
		    settings.ClearOnOpen = chkClearOnOpen.Checked;
			settings.Save();
    }

    /// <summary> Populate the form's controls with default settings. </summary>
    private void InitializeControlValues()
    {
			cmbBaudRate.Text = settings.BaudRate.ToString(); 
            cmbPortName.Items.Clear();
            FamSerialComm.EnumerateComPorts();
            int nCount = FamSerialComm.FriendlyNameComPorts.Count;
            int nSelected = 0;
            if (nCount > 0)
            {
                string[] strFriendlyNames = new string[nCount];
                m_strComPorts = new string[nCount];
                FamSerialComm.FriendlyNameComPorts.Keys.CopyTo(m_strComPorts, 0);
                FamSerialComm.FriendlyNameComPorts.Values.CopyTo(strFriendlyNames, 0);
                for (int i = 0; i < nCount; i++)
                {
                    cmbPortName.Items.Add(m_strComPorts[i] + " - " + strFriendlyNames[i]);
                    if (settings.PortName != null)
                        if (settings.PortName.CompareTo(m_strComPorts[i]) == 0)
                            nSelected = i;
                }
            }
            else
            {
                MessageBox.Show(this, "There are no COM Ports detected on this computer.\nPlease install a COM Port and restart this app.", "No COM Ports Installed", MessageBoxButtons.OK, MessageBoxIcon.Error);
                this.Close();
            }

            cmbPortName.SelectedIndex = nSelected;

			chkClearOnOpen.Checked = settings.ClearOnOpen;

    }

    /// <summary> Enable/disable controls based on the app's current state. </summary>
    private void EnableControls()
    {
      // Enable/disable controls based on whether the port is open or not
        cmbBaudRate.Enabled = cmbPortName.Enabled = !comport.IsOpen;

      if (comport.IsOpen)
      {
          comport.DiscardInBuffer();
          btnOpenPort.Text = "&Close Port";
      }
      else btnOpenPort.Text = "&Open Port";
    }

 
    /// <summary> Log data to the terminal window. </summary>
    /// <param name="msgtype"> The type of message to be written. </param>
    /// <param name="msg"> The string containing the message to be shown. </param>
    private void Log(LogMsgType msgtype, string msg)
    {
      rtfTerminal.Invoke(new EventHandler(delegate
      {
        rtfTerminal.SelectedText = string.Empty;
        rtfTerminal.SelectionFont = new Font(rtfTerminal.SelectionFont, FontStyle.Bold);
        rtfTerminal.SelectionColor = LogMsgTypeColor[(int)msgtype];
        rtfTerminal.AppendText(msg);
        rtfTerminal.ScrollToCaret();
      }));
    }


    //for showing RAW image on the screen
    private void ShowImage(int BitmapWidth, int BitmapHeight, byte[] pRawImage)
    {
        MyBitmapFile myFile = new MyBitmapFile(BitmapWidth, BitmapHeight, pRawImage);
        MemoryStream BmpStream = new MemoryStream(myFile.BitmatFileData);
        Bitmap Bmp = new Bitmap(BmpStream);
        pictureBox1.Image = Bmp;
    }
    
    delegate void SetProgressBarCallback(byte Action);  //0 - Invisible, 1 - Visible, 2 - PerformStep
    private void SetProgressbar(byte Action)
    {
        // Do not change the state control during application closing.
        if (m_bExit)
            return;
        if (this.progressBar1.InvokeRequired)
        {
            if (!m_bExit)
            {
                SetProgressBarCallback d = new SetProgressBarCallback(this.SetProgressbar);
                this.Invoke(d, new object[] { Action });
            }
        }
        else
        {
            if (Action == 0)
            {
                progressBar1.Value = 0;
                progressBar1.Visible = false;
            }
            else if (Action == 1)
            {
                progressBar1.Maximum = TotalDataSize;
                progressBar1.Visible = true;
            }
            else
            {
                progressBar1.Value = DataReceiveCount;
                progressBar1.PerformStep();
            }
        }
    }


    //check checksum and calculating data size from the command received from FS28
    private bool Cmd_Checksum_DataLen_Cal(byte[] cmd, ref int size)
    {
        int image_data_Size = 0;
        int CheckSum = 0;
        for (int i = 0; i < 11; i++)
        {
            CheckSum += cmd[i];
        }
        CheckSum = (CheckSum & 0xFF);        
        if (CheckSum != cmd[11])
            return false;

        /* Data Size
         * RAW image: fixed at 153602 bytes: Data (153600 bytes)+ checksum (1 byte) + stop byte (0x0D).
         * WSQ image: not fixed.
         * Futronic Sample: fixed at 666 bytes: Data (664 bytes)+ checksum (1 byte) + stop byte (0x0D).
         * ANSI Sample: not fixed.
         * ISO Sample: not fixed.
         */
        for (int i = 3; i > 0; i--) 
		{
            image_data_Size |= cmd[6 + i];
            image_data_Size = (image_data_Size << 8);
		}
        image_data_Size |= cmd[6];
        image_data_Size += 2; // add 2 becuase for last 2 extra bytes which include checksum (1 byte) and stop byte (0x0D).
        size = image_data_Size;
        return true;
    }

    #endregion

    #region Respond to MCU

    private void SendRsp2FS28(byte cmd, byte len, byte flag)
    {
        byte checksum = 0;
        Rsp2FS28[1] = cmd;
        Rsp2FS28[6] = len;
        Rsp2FS28[6] = 0xFF;
        Rsp2FS28[10] = flag; //0x40 for ok, 0x41 for fail
        for (int i = 0; i < 11; i++)
            checksum += Rsp2FS28[i];
        Rsp2FS28[11] = checksum;
        comport.Write(Rsp2FS28,0,13);
    }

    #endregion

    #region Event Handlers


    private void frmTerminal_Shown(object sender, EventArgs e)
    {
      Log(LogMsgType.Normal, String.Format("Application Started at {0}\n", DateTime.Now));
    }
    private void frmTerminal_FormClosing(object sender, FormClosingEventArgs e)
    {
        SaveSettings();
      
    }

    private void cmbBaudRate_Validating(object sender, CancelEventArgs e)
    { int x; e.Cancel = !int.TryParse(cmbBaudRate.Text, out x); } 


    private void btnOpenPort_Click(object sender, EventArgs e)
    {
			bool error = false;

      // If the port is open, close it.
            if (comport.IsOpen)
            {
                comport.Close();
            }
            else
            {
                // Set the port's settings
                comport.BaudRate = int.Parse(cmbBaudRate.Text);
                comport.DataBits = 8;
                comport.StopBits = (StopBits)Enum.Parse(typeof(StopBits), "1");
                comport.Parity = (Parity)Enum.Parse(typeof(Parity), "0");
                comport.PortName = m_strComPorts[cmbPortName.SelectedIndex];
                for (int i = 0; i < 3; i++)
                {
                    try
                    {
                        // Open the port
                        comport.Open();
                    }
                    catch (UnauthorizedAccessException) { error = true; }
                    catch (IOException) { error = true; }
                    catch (ArgumentException) { error = true; }

                    if (error)
                    {
                        error = false;
                        if (i == 2)
                        {
                            MessageBox.Show(this, "Could not open the COM port.  Most likely it is already in use, has been removed, or is unavailable.", "COM Port Unavalible", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                            break;
                        }
                    }
                    else
                        break;
                }
            }

      // Change the state of the form's controls
      EnableControls();

      // If the port is open, send focus to the send data box
			if (comport.IsOpen)
			{
                enterCmdMode();
				if (chkClearOnOpen.Checked) ClearTerminal();
             }
    }

    //COM port recive commands or WSQ/RAW image or Futronic/ANSI/ISO sample from FS28
    private void port_DataReceived(object sender, SerialDataReceivedEventArgs e)
    {
        if (!comport.IsOpen) return; // If the com port has been closed, do nothing

        // Obtain the number of bytes waiting in the port's buffer
        int bytes = comport.BytesToRead;        
 
        // Create a byte array buffer to hold the incoming data
        byte[] buffer = new byte[bytes];

        // Read the data from the port and store it in our buffer
        comport.Read(buffer, 0, bytes);

        if (!DataMode)// if not DataMode, then it should be Command mode which is reciving the 13 bytes command from FS28
        {
            if( bytes < 13 )
                return;
            byte[] command = new byte[13];
            int tempsize = 0;
            Array.Copy( buffer, command, 13 );
            if (!Cmd_Checksum_DataLen_Cal(command, ref tempsize))//calculate the image size and check whether checksum is match or not.
            {
                
                comport.DiscardOutBuffer();
                SendRsp2FS28(0xA0, 0x00, 0x41); //reply fail
                comport.DiscardInBuffer();
                Log(LogMsgType.Incoming, String.Format("\nWrong CheckSum\n"));
                return;
            }
            if (buffer[1] == 0x36) // 0x36 command means FS28 start to convert WSQ
            {
                Log(LogMsgType.Incoming, "\nGot convert WSQ cmd\n");
                startReceiveTime = System.Environment.TickCount;
                comport.DiscardOutBuffer(); //clear the out buffer incase there is any data fail to send to FS28
                comport.DiscardInBuffer(); //clear the in buffer incase there is any data fail to receive from FS28
                SendRsp2FS28(0xA0, 0x00, 0x40);//reply ok
                return;
            }
            else if (buffer[1] == 0x0F) //0x0F command means FS28 start to download WSQ
            {
                TotalDataSize = tempsize;
                Log(LogMsgType.Incoming, String.Format("\nGot download WSQ cmd. Size is {0:d} byte\n", TotalDataSize));
                Log(LogMsgType.Incoming, String.Format("Convert WSQ cmd time is {0:d} (ms)\n", (System.Environment.TickCount - startReceiveTime)));

                SendRsp2FS28(0xA0, 0x00, 0x40);//reply ok
                ImageType = ImageTypes.WSQ;
            }
            else if (buffer[1] == 0x44) //0x44 command means FS28 start to download RAW image, the RAW image size is always 153602 byte
            {
                ImageType = ImageTypes.RawImage;
                TotalDataSize = tempsize;
                Log(LogMsgType.Incoming, String.Format("\nGot download RAW cmd. Size is {0:d} byte\n", TotalDataSize));
                comport.DiscardOutBuffer(); //clear the out buffer incase there is any data fail to send to FS28
                comport.DiscardInBuffer(); //clear the in buffer incase there is any data fail to receive from FS28
                SendRsp2FS28(0xA0, 0x00, 0x40);//reply ok
            }
            else if (buffer[1] == 0x4D) //0x4D command means FS28 start to download Futronic/ANSI/ISO sample.
            {
                
                TotalDataSize = tempsize;
                if (buffer[10] == 0x00) // Futronic Sample fixed at 666bytes
                {
                    TotalDataSize = 666;// make it backward compatiable with old FS28 firmware(below AVR v5.2.0) which will not tell the data size in the command
                    ImageType = ImageTypes.FutronicSample;
                    Log(LogMsgType.Incoming, String.Format("\nGot download Futronic Sample cmd. Size is {0:d} byte\n", TotalDataSize));
                }
                else if (buffer[10] == 0x28) // ANSI Sample, no fixed length
                {
                    ImageType = ImageTypes.AnsiSample;
                    Log(LogMsgType.Incoming, String.Format("\nGot download ANSI Sample cmd. Size is {0:d} byte\n", TotalDataSize));
                }
                else if (buffer[10] == 0x38) // ISO Sample, no fixed length
                {
                    ImageType = ImageTypes.IsoSample;
                    Log(LogMsgType.Incoming, String.Format("\nGot download ISO Sample cmd. Size is {0:d} byte\n", TotalDataSize));
                }
                else
                { } //should never reach here.
                comport.DiscardOutBuffer(); //clear the out buffer incase there is any data fail to send to FS28
                comport.DiscardInBuffer(); //clear the in buffer incase there is any data fail to receive from FS28
                SendRsp2FS28(0xA0, 0x00, 0x40);//reply ok
            }
            else
            {
                Log(LogMsgType.Incoming, "\nUnknown command!\n");
                comport.DiscardOutBuffer();
                SendRsp2FS28(0xA0, 0x00, 0x41); //reply fail
                comport.DiscardInBuffer();
                return;
            }
            if (TotalDataSize > 153602)
            {
                Log(LogMsgType.Incoming, "\nTotal data size is too large!\n");//should never reach here!
                return;
            }
            pictureBox1.Image = null;
            startReceiveTime = System.Environment.TickCount;
            Array.Clear(DataReceived, 0, 153602);

            Array.Copy(buffer, 13, DataReceived, 0, bytes-13);//copy the remaining bytes in buffer which is part of the image or sample data and have to be store in DataReceived.
            SetProgressbar(1);

            DataReceiveCount = bytes - 13; //Apart from the 13 byte command, the remaing bytes are part of the WSQ/RAW image data
            DataMode = true;  
            comport.ReceivedBytesThreshold = 1;
            bImgRxComplete = false; //inicating Image is not competely received.
            EnableTimerInvoke(true); // start counting timeout to receve the RAW image or WSQ data.

        }
        else //DataMode which is receving WSQ/RAW image or sample data from FS28, no command anymore.
        {
            buffer.CopyTo(DataReceived, DataReceiveCount);
            DataReceiveCount += bytes;
        }

        SetProgressbar(2);//update the progress bar.
        if (DataReceiveCount >= TotalDataSize)// finish reciving all the WSQ/RAW image data
        {
            EnableTimerInvoke(false); //stop the timeout counter as WSQ/RAW or sample has beeen completely received
            Log(LogMsgType.Incoming, String.Format("Finish receving data,time used {0:d} (ms)\n", System.Environment.TickCount - startReceiveTime));
            SendRsp2FS28(0xA0, 0x00, 0x40);//reply ok, finished receiving the RAW image or WSQ or sample.
            bImgRxComplete = true;

            switch (ImageType)
            {
                case ImageTypes.RawImage: //show RAW image
                    ShowImage(320, 480, DataReceived);
                    break;
                case ImageTypes.WSQ: //convert WSQ image to RAW and show it
                    FTRIMGPARMS ImgParm;
                    ImgParm = new FTRIMGPARMS();
                    ImgParm.WSQ_size = (uint)(TotalDataSize - 2);
                    sizeWsqImg = ImgParm.WSQ_size;
                    if (FtrWSQ.ftrWSQ_GetImageParameters(DataReceived, ref ImgParm) == 0)
                    {
                        Log(LogMsgType.Incoming, "Failed to call ftrWSQ_GetImageParameters\n");
                        return;
                    }
                    ConvertedRawImage = new byte[ImgParm.RAW_size];
                    if (FtrWSQ.ftrWSQ_ToRawImage(DataReceived, ref ImgParm, ConvertedRawImage) == 0)
                    {
                        Log(LogMsgType.Incoming, "Failed to call ftrWSQ_ToRawImage\n");
                        return;
                    }
                    ShowImage((int)ImgParm.Width, (int)ImgParm.Height, ConvertedRawImage);
                    break;
                case ImageTypes.FutronicSample: // convert FS28 sample to Host SDK sample and do identification
                    ConvertToHostSample(DataReceived);                    
                    Log(LogMsgType.Incoming, "Futronic Sample downloaded\n");
                    // identify using Futronic's proprietary SDK...
                    Identify identifySample = new Identify();
                    identifySample.StartIdentify(HostSample);
                    break;
                case ImageTypes.AnsiSample:
                    sizeAnsiIso =  DataReceiveCount;
                    String fsPathAnsi = Environment.CurrentDirectory + @"\default.ansi";
                    FileStream AnsiFS = new FileStream(fsPathAnsi, FileMode.Create);
                    try
                    {
                        AnsiFS.Write(DataReceived, 0, sizeAnsiIso); //ANSI sample
                    }
                    catch(System.Exception)
                    {
                        AnsiFS.Close();
                        Log(LogMsgType.Incoming, "ANSI Sample dowmloaded but not saved\n");
                        break;
                    }
                    AnsiFS.Close();
                    Log(LogMsgType.Incoming, "ANSI Sample downloaded and saved");
                    break;
                case ImageTypes.IsoSample:
                    sizeAnsiIso = DataReceiveCount;
                    String fsPathIso = Environment.CurrentDirectory + @"\default.iso1";
                    FileStream IsoFS = new FileStream(fsPathIso, FileMode.Create);
                    try
                    {
                        IsoFS.Write(DataReceived, 0, sizeAnsiIso); //ANSI sample
                    }
                    catch(System.Exception)
                    {
                        IsoFS.Close();
                        Log(LogMsgType.Incoming, "ANSI Sample dowmloaded but not saved\n");
                        break;
                    }
                    IsoFS.Close();
                    Log(LogMsgType.Incoming, "ISO Sample downloaded and saved\n");
                    break;
                default:
                    break;

            }
            enterCmdMode();//return to Command mode to receive 13 byte command from FS28

        }
       

    }
      /****
       * convert the sample received from FS28 to the host format for Futronic's proprietary SDK
      ****/
      private void ConvertToHostSample(byte[] FamSample)
      {
          HostSample[0] = 0x9d;
          HostSample[1] = 0x02;
          HostSample[2] = 0x02;
          HostSample[3] = 0x02;
          HostSample[4] = 0x00;
          Array.Copy(FamSample, 0, HostSample, 5, 664);
      }


    #endregion



		private void btnClear_Click(object sender, EventArgs e)
		{
			ClearTerminal();
		}

		private void ClearTerminal()
		{
			rtfTerminal.Clear();
		}

       

        private void btnSaveImage_Click(object sender, EventArgs e)
        {
            if (!bImgRxComplete)
            {
            MessageBox.Show(this, "No image received");
            return;
            }

            switch (ImageType)
            {
                case ImageTypes.RawImage:
                    saveFileDialog1.Filter = "bmp files (*.bmp)|*.bmp|raw files (*.raw)|*.raw";
                    break;
                case ImageTypes.WSQ:
                    saveFileDialog1.Filter = "bmp files (*.bmp)|*.bmp|raw files (*.raw)|*.raw|wsq files (*.wsq)|*.wsq";
                    break;
                case ImageTypes.FutronicSample:
                    saveFileDialog1.Filter = "binary files (*.bin)|*.bin";
                    break;
                case ImageTypes.AnsiSample:
                    saveFileDialog1.Filter = "binary files (*.ansi)|*.ansi";
                    break;
                case ImageTypes.IsoSample:
                    saveFileDialog1.Filter = "binary files (*.iso1)|*.iso1";
                    break;
                default:
                    break;
            }

            if (saveFileDialog1.ShowDialog() == DialogResult.OK)
            {
                FileStream file = new FileStream(saveFileDialog1.FileName, FileMode.Create);
                if (saveFileDialog1.FilterIndex == 1)   //bitmap
                {
                    MyBitmapFile myFile;

                    switch (ImageType)
                    {
                        case ImageTypes.RawImage:
                            myFile = new MyBitmapFile(320, 480, DataReceived);
                            file.Write(myFile.BitmatFileData, 0, myFile.BitmatFileData.Length);
                            break;
                        case ImageTypes.WSQ:
                            myFile = new MyBitmapFile(320, 480, ConvertedRawImage);
                            file.Write(myFile.BitmatFileData, 0, myFile.BitmatFileData.Length);
                            break;
                        case ImageTypes.FutronicSample:
                            file.Write(HostSample, 0, 669); //PC sample
                            //file.Write(DataReceived, 0, 666);//FAM sample
                            break;
                        case ImageTypes.AnsiSample:
                            file.Write(DataReceived, 0,sizeAnsiIso); //ANSI sample
                            break;
                        case ImageTypes.IsoSample:
                            file.Write(DataReceived, 0,sizeAnsiIso); //ISO sample
                            break;
                        default:
                            break;
                    }
                }
                else if (saveFileDialog1.FilterIndex == 2)  //binary
                {

                    switch (ImageType)
                    {
                        case ImageTypes.RawImage:
                            file.Write(DataReceived, 0, 153600);
                            break;
                        case ImageTypes.WSQ:
                            file.Write(ConvertedRawImage, 0, 153600);
                            break;
                        default:
                            break;
                    }
                }
                else if (saveFileDialog1.FilterIndex == 3)  //wsq
                {
                    switch (ImageType)
                    {
                        case ImageTypes.WSQ:
                            if (sizeWsqImg > 0)
                                file.Write(DataReceived, 0, (int)sizeWsqImg);
                            break;
                        default:
                            break;
                    }
                }
                file.Close();
            }
        }

        private void ImgRecTmr_Tick(object sender, EventArgs e)// timeout for receiving WSQ or RAW image.
        {
            if (comport.IsOpen)
            {
                comport.DiscardOutBuffer();
                SendRsp2FS28(0xA0, 0x00, 0x41); //reply fail
                comport.DiscardInBuffer();
            }
            enterCmdMode();
        }

        private void enterCmdMode()// go to command mode to receive 13 byte commands from FS28
        {

            
            //reset variable value
            DataReceiveCount = 0;
            TotalDataSize = 0;
            DataMode = false;
            EnableTimerInvoke(false);
            if (comport.IsOpen)
            {
                comport.ReceivedBytesThreshold = 13;
                comport.DiscardInBuffer();
            }
            SetProgressbar(0);
        }  

	}
}